#include "SATinstance.h"
#include "global.h"

#include <stdio.h>
#include <math.h>
#include <time.h>

#include <sys/times.h>
#include <limits.h>
#include <stdlib.h>

#include <set>
#include <map>
#include <algorithm>
#include <iostream>
#include <fstream>
#include <sstream>
#include <cstring>
#include <vector>
#include <cassert>
#include "stopwatch.h"

#define MAX(X,Y) ((X) > (Y) ? (X) : (Y))
#define MIN(X,Y) ((X) > (Y) ? (Y) : (X))
#define ABS(X) ((X) > 0 ? (X) : -(X))

#define positive(X) ((X) > 0)
#define negative(X) ((X) < 0)

#define RESERVED_VALUE (-512)


const char *SATinstance::badFeatNames[] = { "Pre-featuretime", "Basic-featuretime", "KLB-featuretime", "DIAMETER-featuretime", "cl-featuretime", "sp-featuretime", "unit-featuretime", "lobjois-featuretime", "ls-featuretime", "UNARY", "POSNEG-RATIO-CLAUSE-max" , "POSNEG-RATIO-CLAUSE-min"};
const int SATinstance::numBadFeats = 12;

// == THIS IS HORRIBLE STYLE! But the easiest way to interface 
// == with ubcsat for now
SATinstance* currInstanceForUBC=0;

void writeFeature(const char* name, double val)
{
  currInstanceForUBC->writeFeature(name, val);
}


SATinstance::SATinstance(const char* filename, long _seed) 
  :ignoreBadFeats(true),
   seed(_seed)
{
  currInstanceForUBC = this;

  ifstream infile(filename);
  if (!infile) {
    fprintf(stderr, "c Error: Could not read from input file %s.\n", filename);
    exit(1);
  }

  inputFileName = (char *)filename;
  char chbuf;
  char strbuf[1024];
  infile.get(chbuf);
  while (chbuf != 'p') {
    //    infile.getline(strbuf, 100);
    infile.ignore(1000, '\n');
    infile.get(chbuf);
    if(!infile)
      {
	fprintf(stderr, "c ERROR: Premature EOF reached in %s\n", filename);
	exit(1);
      }
  }

  infile >> strbuf; // "cnf"
  if( strcmp(strbuf, "cnf")!=0 )
    {
      fprintf(stderr, "c Error: Can only understand cnf format!\n");
      exit(1);
    }
  // == TODO: During parsing should really skip comment lines....
  // == TODO: Parser should really check for EOF

  infile >> numVars >> numClauses;

  clauses = new int*[numClauses];
  clauseStates = new clauseState[numClauses];
  clauseLengths = new int[numClauses];

  negClausesWithVar = new vector<int>[numVars+1];
  posClausesWithVar = new vector<int>[numVars+1];

  numActiveClausesWithVar = new int[numVars+1];
  numBinClausesWithVar = new int[numVars+1];

  for (int i=1; i<=numVars; i++) {
    numActiveClausesWithVar[i] = 0;
    numBinClausesWithVar[i] = 0;
  }

  int *lits = new int[numVars+1];
  
  // read stuff into data structure. 
  // take care of all data structure
  // clauseLengths,  unitClauses, numBinClausesWithVar
  // negClausesWithVar, posClausesWithVar, numActiveClausesWithVar
  for (int clauseNum=0; clauseNum<numClauses; clauseNum++) {
    int numLits = 0;  // not including 0 terminator
    if(!infile)
      {
	fprintf(stderr, "c ERROR: Premature EOF reached in %s\n", filename);
	exit(1);
      }

    infile >> lits[numLits];
    while (lits[numLits] != 0){  
      infile >> lits[++numLits];
    }
   
    /* test if some literals are redundant and sort the clause */
    bool tautology = false;
    for (int i=0; i<numLits-1; i++) {
      int tempLit = lits[i];
      for (int j=i+1; j<numLits; j++) {
    	if (ABS(tempLit) > ABS(lits[j])) {
// this is sorting the literals	 
            int temp = lits[j];
            lits[j] = tempLit;
	         tempLit = temp;
	     } else if (tempLit == lits[j]) {
	       lits[j--] = lits[--numLits];
	  printf("c literal %d is redundant in clause %d\n", tempLit, clauseNum);
	      } else if (ABS(tempLit) == ABS(lits[j])) {
	        tautology = true;
//	  printf("c Clause %d is tautological.\n", clauseNum);
//	  break;
	      }
      }
      if (tautology) break;
      else lits[i] = tempLit;
    }              
  	
    if (!tautology) {
      clauseLengths[clauseNum] = numLits;
      clauses[clauseNum] = new int[numLits+1];
      clauseStates[clauseNum] = ACTIVE;

      if (numLits == 1)
	      unitClauses.push(clauseNum);
      else if (numLits == 2)
	      for (int i=0; i<numLits; i++)
	          numBinClausesWithVar[ABS(lits[i])]++;

          for (int litNum = 0; litNum < numLits; litNum++) {
	           if (lits[litNum] < 0)
	               negClausesWithVar[ABS(lits[litNum])].push_back(clauseNum);
	           else
	               posClausesWithVar[lits[litNum]].push_back(clauseNum);
               numActiveClausesWithVar[ABS(lits[litNum])]++;
               clauses[clauseNum][litNum] = lits[litNum];
               }
      clauses[clauseNum][numLits] = 0;
    } else {
      clauseNum--;
      numClauses--;
    }
//    printf("%d, %d, %d, %d %d\n", clauses[clauseNum][0],clauses[clauseNum][1],clauses[clauseNum][2],clauseNum, numClauses);
  }
  delete[] lits;
  numActiveClauses = numClauses;

// remove some redandant variables
// prepar data sturcuture: varStates
  varStates = new varState[numVars+1];
  numActiveVars = numVars;
  for (int i=1; i<=numVars; i++) {
    if (numActiveClausesWithVar[i]  == 0) {
      varStates[i] = IRRELEVANT;
      numActiveVars--;
    } else
      varStates[i] = UNASSIGNED;
  }
  
// before doing anything first do a round of unit propogation to remove all the 
// unit clasues
  int dummy1, dummy2;
  unitprop(dummy1, dummy2);

  test_flag = new int[numVars+1];
  indexCount = 0;    

  if (seed == 0)
    seed=(long)time(NULL);
      
  srand(seed);
}

SATinstance::~SATinstance() {
  delete[] varStates;
  for (int i=0; i<numClauses; i++)
    delete[] clauses[i];
  delete[] clauses;
  delete[] clauseStates;
  delete[] clauseLengths;

  delete[] negClausesWithVar;
  delete[] posClausesWithVar;

  delete[] numActiveClausesWithVar;
  delete[] numBinClausesWithVar;

  delete[] test_flag;
}

inline vector<int> &SATinstance::clausesWithLit(int lit) {
  if (positive(lit))
    return posClausesWithVar[lit];
  else
    return negClausesWithVar[-lit];
}

bool SATinstance::reduceClauses(int lit, int &numClausesReduced, int &numVarsReduced) {

  // "remove" vars from inconsistent clauses
  for (int i=0; i<(int)clausesWithLit(-lit).size(); i++) {
    int clause = clausesWithLit(-lit)[i];
    if (clauseStates[clause] == ACTIVE) {
      reducedClauses.push(clause);
      numClausesReduced++;

      clauseLengths[clause]--;
      if (clauseLengths[clause] == 2)
    	for (int i=0; clauses[clause][i] != 0; i++)
	     numBinClausesWithVar[ABS(clauses[clause][i])]++;
        else if (clauseLengths[clause] == 1) {
	         for (int i=0; clauses[clause][i] != 0; i++)
	             numBinClausesWithVar[ABS(clauses[clause][i])]--;
	             unitClauses.push(clause);

      } else if (clauseLengths[clause] == 0)
	return false;
    }
  }

  // satisfy consistent clauses
  for (int i=0; i<(int)clausesWithLit(lit).size(); i++) {
    int clause = clausesWithLit(lit)[i];
    if (clauseStates[clause] == ACTIVE) {
	
      clauseStates[clause] = PASSIVE;
      reducedClauses.push(clause);
      numActiveClauses--;

      int j=0;
      int otherVarInClause = ABS(clauses[clause][j]);
      while (otherVarInClause != 0) {
    	numActiveClausesWithVar[otherVarInClause]--;
	    if (clauseLengths[clause] == 2)
	        numBinClausesWithVar[otherVarInClause]--;

	// is the var now irrelevant (active, but existing in no clauses)?
	if (numActiveClausesWithVar[otherVarInClause] == 0 &&
	    varStates[otherVarInClause] == UNASSIGNED) {
	    varStates[otherVarInClause] = IRRELEVANT;
        reducedVars.push(otherVarInClause);
        numActiveVars--;
	  
	  numVarsReduced++;
	}

	j++;
	otherVarInClause = ABS(clauses[clause][j]);
      }
      numClausesReduced++;
    }
  }
  return true;
}


bool SATinstance::unitprop(int &numClausesReduced, int &numVarsReduced) {

  bool consistent = true;

  while (!unitClauses.empty() && consistent) {
    int clauseNum = unitClauses.top();
    unitClauses.pop();

    if (clauseStates[clauseNum] != ACTIVE) continue;

    int litNum = 0;
    while (varStates[ABS(clauses[clauseNum][litNum])] != UNASSIGNED) {
      litNum++;
    }

    // assertions are our friends!
    assert (clauseLengths[clauseNum] == 1);

    int lit = clauses[clauseNum][litNum];

    varStates[ABS(lit)] = positive(lit) ? TRUE_VAL : FALSE_VAL;
    reducedVars.push(ABS(lit));
    numActiveVars--;
    numVarsReduced++;

    consistent &= reduceClauses(lit, numClausesReduced, numVarsReduced);
  }

  return consistent;
}


inline void p(const char *in) {
#ifdef DEBUG
  printf ("%s\n",in);
  fflush(stdout);
#else
#endif
}

int SATinstance::computeFeatures() {

  Stopwatch sw;
  sw.Start();
    
  //fprintf(stderr, "Computing features. Prefix %s endpref\n", featurePrefix);
  // node degree stats for var-clause graph
  int *var_array = new int[numVars+1];
  int *var_graph = new int[numVars+1];
  bool *var_graph_found = new bool[numVars+1];
  double *var_graph_norm = new double[numVars+1];
  int *horny_var = new int[numVars+1];
  double *horny_var_norm = new double[numVars+1];
  double *var_array_norm = new double[numVars+1];
  int *clause_array = new int[numClauses];
  double *clause_array_norm = new double[numClauses];
  int *pos_in_clause = new int[numClauses];
  int *neg_in_clause = new int[numClauses];
  double *pos_frac_in_clause = new double[numClauses];
  int *pos_var = new int [numVars+1];
  int *neg_var = new int [numVars+1];
  double *pos_frac_per_var = new double [numVars+1];
  int unary=0, binary=0, trinary=0;
  int horn_clauses = 0;
  int t, tt;
    
  // initialize
  for (t=1; t<=numVars; t++) 
    {
      var_array[t] = 0;
      pos_var[t] = 0;
      neg_var[t] = 0;
      horny_var[t] = 0;
      var_array_norm[t] = RESERVED_VALUE;
      pos_frac_per_var[t] = RESERVED_VALUE;
    }
    
  for (t=0;t<numClauses;t++)
    {
      clause_array[t] = (int)RESERVED_VALUE;
      clause_array_norm[t] = RESERVED_VALUE;
      pos_in_clause[t] = (int)RESERVED_VALUE;
      neg_in_clause[t] = (int)RESERVED_VALUE;
      pos_frac_in_clause[t] = RESERVED_VALUE;
    }
    
  p("c Go through clauses...");
  writeFeature("nvars",(double)numActiveVars);
  writeFeature("nclauses",(double)numActiveClauses);
  writeFeature("vars-clauses-ratio",((double)numActiveVars)/(double)numActiveClauses);

  // go through all the clauses
  // What we get from here is 
  // clause_array : number of lierals
  // pos_in_clause/neg_in_clause
  // var_array: number of cluses contain this variable
  // pos_var/neg_var
  int *clause, lit;
  t=0;
  for (clause = firstClause(); clause != NULL; clause = nextClause()) 
    {
      // initialize 
      clause_array[t] = 0;
      pos_in_clause[t] = 0;
      neg_in_clause[t] = 0;
        
      for (lit = firstLitInClause(clause); lit != 0; lit = nextLitInClause()) 
	{
	  clause_array[t]++;
	  var_array[ABS(lit)]++;
            
	  if (positive(lit)) 
	    {
	      pos_in_clause[t]++;
	      pos_var[ABS(lit)]++;
	    }
	  else 
	    {
	      neg_in_clause[t]++;
	      neg_var[ABS(lit)]++;
	    }
	}
        
   // may be this is a bad name for this. 
   // basically, it compute the bias for the assignment     
   // do we need say anything for cluase_array[t]=0
   // this should not happened
      if(clause_array[t]!=0)
	     pos_frac_in_clause[t] = 2.0 * fabs(0.5 - (double) pos_in_clause[t] / ((double)pos_in_clause[t] + (double)neg_in_clause[t]));
      else
	{
	  pos_frac_in_clause[t]=RESERVED_VALUE;
	  //	  fprintf(stderr, "L %d clause %d empty\n", featureLevel, t);
	}
        
      // cardinality
      switch(clause_array[t]) 
	{
	case 1: unary++; break;
	case 2: binary++; break;
	case 3: trinary++; break;
	}

      // NOTE: isn't neg_in_clause <= 1 also horny? GMA
      // this is really not make sense. by switching pos/neg, you can get different horn clause  
      // horn clause
      if (pos_in_clause[t] <= 1)
	{
	  for (lit = firstLitInClause(clause); lit != 0; lit = nextLitInClause()) 
	    horny_var[ABS(lit)]++;
	  horn_clauses++;
	}        
      // normalize
      clause_array_norm[t] = (double) clause_array[t] / (double) numActiveVars;   
      // increment clause index
      t++;
    }
  //  fprintf(stderr, "Level %d: Went through %d clauses\n", featureLevel, t);
  
   // positive ratio in clauses
  writeStats(pos_frac_in_clause,numClauses,"POSNEG-RATIO-CLAUSE");
  writeFeature("POSNEG-RATIO-CLAUSE-entropy",array_entropy(pos_frac_in_clause,numClauses,100,1));
    // clause side in the bipartite graph
  writeStats(clause_array_norm,numClauses,"VCG-CLAUSE");
  writeFeature("VCG-CLAUSE-entropy",array_entropy(clause_array,numClauses,numActiveVars+1));
   // cardinality of clauses
  writeFeature("UNARY",(double)unary/(double)numActiveClauses);
  writeFeature("BINARY+",(double)(unary+binary)/(double)numActiveClauses);
  writeFeature("TRINARY+",(double)(unary+binary+trinary)/(double)numActiveClauses);
  
  p("c Go through variables...");

  // Go through the variables
  for (t=1; t <= numVars; t++) 
    {
      if (varStates[t] != UNASSIGNED || var_array[t]==0)  // do we still want the second part?
	{
	  var_graph[t] = (int)RESERVED_VALUE;
	  var_array_norm[t] = RESERVED_VALUE;
	  var_graph_norm[t] = RESERVED_VALUE;
	  horny_var[t] = (int)RESERVED_VALUE;
	  horny_var_norm[t] = RESERVED_VALUE;
	  var_array[t] = (int)RESERVED_VALUE;
	  pos_var[t] = (int)RESERVED_VALUE;
	  neg_var[t] = (int)RESERVED_VALUE;
	  pos_frac_per_var[t] = RESERVED_VALUE;
	  continue;
	}
        
      // calculate and normalize
      pos_frac_per_var[t] = 2.0 * fabs(0.5 - (double)pos_var[t] / ((double)pos_var[t] + (double)neg_var[t]));
      var_array_norm[t] = (double) var_array[t] / (double) numActiveClauses;
      horny_var_norm[t] = (double) horny_var[t] / (double) numActiveClauses;
    }
    
   // variable side in the bipartite graph
  writeStats(var_array_norm+1, numActiveVars, "VCG-VAR");
  writeFeature("VCG-VAR-entropy",array_entropy(var_array+1,numActiveVars,numActiveClauses+1));

  /* == DEBUG:  
     fprintf(stderr, "c L %d: %lf %lf %lf %lf\n", featureLevel, array_min(clause_array_norm, NB_CLAUSE), array_max(clause_array_norm, NB_CLAUSE), mean(clause_array_norm, NB_CLAUSE), stdev(clause_array_norm, NB_CLAUSE, mean(clause_array_norm, NB_CLAUSE)));
     for(t=0; t<NB_CLAUSE; t++)
     {
     fprintf(stderr, "c L %d clause[%d]:\t", featureLevel, t);
     if(clause_array_norm[t]==RESERVED_VALUE) fprintf(stderr, "RESERVED\n");
     else fprintf(stderr, "c %lf\n", clause_array_norm[t]);
     }
  */

    
  // positive ratio in variables
  writeStatsSTDEV(pos_frac_per_var+1,numActiveVars,"POSNEG-RATIO-VAR");
  writeFeature("POSNEG-RATIO-VAR-entropy",array_entropy(pos_frac_per_var+1,numActiveVars,100,1));
    
  // horn clauses
  writeStats(horny_var_norm+1,numActiveVars,"HORNY-VAR");
  writeFeature("HORNY-VAR-entropy",array_entropy(horny_var+1,numActiveVars,numActiveClauses+1));
  writeFeature("horn-clauses-fraction",(double)horn_clauses / (double)numActiveClauses);
    
    
  // clean up after yourself, you pig!
  delete[] var_array;
  delete[] var_graph;
  delete[] var_graph_norm;
  delete[] horny_var;
  delete[] horny_var_norm;
  delete[] var_array_norm;
  delete[] clause_array;
  delete[] clause_array_norm;
  delete[] pos_in_clause;
  delete[] neg_in_clause;
  delete[] pos_frac_in_clause;
  delete[] pos_var;
  delete[] neg_var;
  delete[] pos_frac_per_var;
  delete[] var_graph_found;
    
    
  p("c Done with base features");

  return FEAT_OK;
}


int *SATinstance::firstClause() {
  currentClause = 0;
  return nextClause();
}

int *SATinstance::nextClause() {
  while (currentClause < numClauses && clauseStates[currentClause] != ACTIVE)
    currentClause++;
    
  return currentClause >= numClauses ? NULL : clauses[currentClause++];
}

int SATinstance::firstLitInClause(int *clause) {
  currentClauseForLitIter = clause;
  currentLit = 0;
  return nextLitInClause();
}

int SATinstance::nextLitInClause() {
  while (currentClauseForLitIter[currentLit] != 0 &&
	 varStates[ABS(currentClauseForLitIter[currentLit])] != UNASSIGNED)
    currentLit++;
    
  if (currentClauseForLitIter[currentLit] == 0)
    return 0;
  else
    return currentClauseForLitIter[currentLit++];
}

// mean.  Skips over values of "RESERVED_VALUE"
inline double SATinstance::mean(int *array, int num)
{
  int total=0, t, reserved_hits = 0;
  for (t=0;t<num;t++) 
    {
      if (array[t] == (int)RESERVED_VALUE)
	{
	  reserved_hits++;
	  continue;
	}
      total += array[t];
    }
    
  if(reserved_hits == num) return 0;
  return (double) total / (double) (num - reserved_hits);
}
inline double SATinstance::mean(double *array, int num)
{
  double total=0.0;
  int t, reserved_hits = 0;
  for (t=0;t<num;t++) 
    {

      if (array[t] == (double)RESERVED_VALUE)
	{
	  reserved_hits++;
	  continue;
	}
      total += array[t];
    }

  if(reserved_hits == num) return 0;
  return total / (double) (num - reserved_hits);
}

// standard deviation
inline double SATinstance::stdev(int *array, int num, double mean)
{   
  double dtotal = 0.0;
  int reserved_hits = 0;
  for (int t=0;t<num;t++) 
    {
      if (array[t] == (int)RESERVED_VALUE)
	{
	  reserved_hits++;
	  continue;
	}
      dtotal += square(array[t] - mean);
    }
  if(reserved_hits == num) return 0;
  return sqrt(dtotal/(double)(num-reserved_hits));
}
inline double SATinstance::stdev(double *array, int num, double mean)
{   
  double dtotal = 0.0;
  int reserved_hits = 0;
  for (int t=0;t<num;t++) 
    {
      if (array[t] == (double)RESERVED_VALUE)
	{
	  reserved_hits++;
	  continue;
	}
      dtotal += square(array[t] - mean);
    }
  if(reserved_hits == num) return 0;
  return sqrt(dtotal/(double)(num-reserved_hits));
}

// min
inline int SATinstance::array_min(int *array, int num)
{
  int m=(1 << 30);
  int reserved_hits = 0;
  for (int t=0;t<num;t++)
    {
      if (array[t] == (int)RESERVED_VALUE) 
	{
	  reserved_hits++;
	  continue;
	}
      m = (m<array[t] ? m : array[t]);
    }
  if(reserved_hits == num) return 0;
  return m;
}

inline double SATinstance::array_min(double *array, int num)
{
  double m=(1 << 30);
  int reserved_hits = 0;
  for (int t=0;t<num;t++)
    {
      if (array[t] == (double)RESERVED_VALUE)
	{
	  reserved_hits++;
	  continue;
	}
      m = (m<array[t] ? m : array[t]);
    }
  if(reserved_hits == num) return 0;
  return m;
}

// max
inline int SATinstance::array_max(int *array, int num)
{
  int m=0;
  int reserved_hits = 0;
  for (int t=0;t<num;t++)
    {
      if (array[t] == (int)RESERVED_VALUE)
	{
	  reserved_hits++;
	  continue;
	}
        
      m = (m>array[t] ? m : array[t]);
    }
  if(reserved_hits == num) return 0;
  return m;
}

inline double SATinstance::array_max(double *array, int num)
{
  double m=0;
  int reserved_hits = 0;
  for (int t=0;t<num;t++)
    {
      if (array[t] == (double)RESERVED_VALUE)
	{
	  reserved_hits++;
	  continue;
	}
      m = (m>array[t] ? m : array[t]);
    }
  if(reserved_hits == num) return 0;
  return m;
}

// entropy
inline double SATinstance::array_entropy(double *array, int num, int vals, int maxval)
{
  int *p = new int[vals+1];
  double entropy = 0.0,pval;
  int t,res=0;
  int idx;

  // initialize
  for (t=0;t<=vals;t++) p[t] = 0;
    
  // make the distribution
  for (t=0;t<num;t++) 
    {
      if (array[t] == (double)RESERVED_VALUE) {res++; continue;}
      idx = (int) floor(array[t] / ((double)maxval/(double)vals));
      //      if (idx > maxval) idx = maxval;
      if ( idx > vals ) idx = vals;
      if ( idx < 0 ) idx = 0;
      p[idx]++;
    }
    
  // find the entropy
  for (t=0; t<=vals; t++)
    {
      if (p[t]) 
	{
	  pval = double(p[t])/double(num-res);
	  entropy += pval * log(pval);
	}
    }
        
  delete[] p;

  return -1.0 * entropy;
}
inline double SATinstance::array_entropy(int *array, int num, int vals)
{
  int *p = new int[vals];
  double entropy = 0.0,pval;
  int t,res=0;

  // initialize
  for (t=0;t<vals;t++) p[t] = 0;
    
  // make the distribution
  for (t=0;t<num;t++) 
    {
      if (array[t] == (int)RESERVED_VALUE) {res++; continue;}
#ifdef DEBUG
      if (array[t] < 0 || array[t] >= vals) {
	printf("c ERROR: bad array indexing in array_entropy!"); exit(1);}
#endif
      p[array[t]]++;
    }
    
  // find the entropy
  for (t=0; t<vals; t++)
    {

      //      fprintf(stderr, "Bin %d/%d: %d\n", t, vals, p[t]);

      if (p[t]) 
	{
	  pval = double(p[t])/double(num-res);
	  entropy += pval * log(pval);
	}
    }

  delete[] p;
  return -1.0 * entropy;
}



// write out node stats
// could these stats all be computed in one pass?

void SATinstance::writeStats(int *array, int num, const char *name)
{
  double m = mean(array,num);
  char buffer[100];
  sprintf(buffer,"%s-mean",name);
  writeFeature(buffer,m);
  sprintf(buffer,"%s-coeff-variation",name);
  double sd = stdev(array, num, m);
  double cv = (fabs(m) < EPSILON && sd < EPSILON ? 0 : sd / m);
  writeFeature(buffer, cv);
  sprintf(buffer,"%s-min",name);
  writeFeature(buffer,(double)array_min(array,num));
  sprintf(buffer,"%s-max",name);
  writeFeature(buffer,(double)array_max(array,num));
}


void SATinstance::writeStats(double *array, int num, const char *name)
{
  double m = mean(array,num);
  char buffer[100];
  sprintf(buffer,"%s-mean",name);
  writeFeature(buffer,m);
  sprintf(buffer,"%s-coeff-variation",name);
  double sd = stdev(array, num, m);
  double cv = (fabs(m) < EPSILON && sd < EPSILON ? 0 : sd / m);
  writeFeature(buffer, cv);
  sprintf(buffer,"%s-min",name);
  writeFeature(buffer,array_min(array,num));
  sprintf(buffer,"%s-max",name);
  writeFeature(buffer,array_max(array,num));
}


void SATinstance::writeStatsSTDEV(int *array, int num, const char *name)
{
  double m = mean(array,num);
  char buffer[100];
  sprintf(buffer,"%s-mean",name);
  writeFeature(buffer,m);
  sprintf(buffer,"%s-stdev",name);
  writeFeature(buffer,stdev(array,num,m));
  sprintf(buffer,"%s-min",name);
  writeFeature(buffer,(double)array_min(array,num));
  sprintf(buffer,"%s-max",name);
  writeFeature(buffer,(double)array_max(array,num));
}

void SATinstance::writeStatsSTDEV(double *array, int num, const char *name)
{
  double m = mean(array,num);
  char buffer[100];
  sprintf(buffer,"%s-mean",name);
  writeFeature(buffer,m);
  sprintf(buffer,"%s-stdev",name);
  writeFeature(buffer,stdev(array,num,m));
  sprintf(buffer,"%s-min",name);
  writeFeature(buffer,array_min(array,num));
  sprintf(buffer,"%s-max",name);
  writeFeature(buffer,array_max(array,num));
}


void SATinstance::print() {
  printf("p cnf %d %d\n", numActiveVars, numActiveClauses);
  for (int clause = 0; clause < numClauses; clause++) {
    if (clauseStates[clause] != ACTIVE) continue;
    //    printf("%d\t", clauseLengths[clause]);
    for (int lit = 0; clauses[clause][lit] != 0; lit++)
      if (varStates[ABS(clauses[clause][lit])] == UNASSIGNED)
	printf("%d\t", clauses[clause][lit]);
    printf("%d", clauseLengths[clause]);
    printf("\n");
  }
  printf("\n");
}


void SATinstance::writeFeature(const char *name, double val) {
  if (ignoreBadFeats)
    for (int i=0; i<numBadFeats; i++){
        if (!strcmp(name, badFeatNames[i])) {
//	printf("c not including %s feature\n", name);
	return;
      }
   }
  string s=string(string(name)); //featurePrefix)+ string(name);
  nameToIndex[s] = indexCount;
  featureNames[indexCount] = strdup(s.c_str());
  featureVals[indexCount] = val;
  indexCount++;
  if(indexCount>=MAX_FEATURES)
    {
      fprintf(stderr, "c TOO MANY FEATURES COMPUTED!\n");
      exit(1);
    }
  // printf("feature %d: %s; val: %f\n", indexCount, name, val);
}

